#include <hidef.h>
#include "comm.h"
#include "derivative.h" 
#include "control_constants.h"
#include "EEEfunctions.h"
#include "ts_functions.h"


byte temp_buffer[] = 0;
byte buffer_index = 0;
byte use_df_coeff;
byte need_to_recal = 0;
byte LCD_aligned = 0;
byte xy_data_toTX[];
byte stay_awake_for_RX = 0;
byte temp;


#pragma DATA_SEG EEE_VARS    //Stored in Flash.  See ".prm" file for configuration

uchar EEE_cal_data[12];     

#pragma DATA_SEG DEFAULT


  
/**************************************************************
*    Function:     transmit(*data)
*    Parameters: *data - pointer to the data that will be transmitted
*    Return:     none
*
*    This function transmits the x-y values out the UART
*      Note that at the slowest baud rate (9600)
*      a significant time may be spent transmiting 4-byte
*      packets. To avoid spending to much time waiting for data
*      to transmit, the interrupt handler is used.  The first byte
*      is started here.  Then the TX buffer empty interrupt handler
*      is started and will handle all remaining bytes.  The interrupt
*      handler can be found in MCUint.c  To avoid communications causing
*      timing issues, the code is defaulted to use 19.2KBaud. 
***************************************************************/

void transmit(byte *data)
{  
    unsigned char Temp;
  
    Temp = SCIS1;             // Read the status byte to ensure the TX Int flag is not set
    SCID = data[0];           // Write the first byte of data
    byte_to_TX = 1;           // Let the interrupt routine know which byte is being transmitted
    SCIC2_TIE = 1;            // Enable the TX Interrupt to handle the rest of the bytes
}


/**************************************************************
*    Function:     Process_RX_Data()
*    Parameters: none
*    Return:     none
*
*    This function processes the information received on the  
*    communication channel.  The data is received in the interrupt
*    routine (see MCUinit.c) and placed in the 'temp_buffer'
*    array.  The first value of the array is the control byte.
*
*    Currently, three commands are implemented:
*    'i': Implement the following coefficients.  This loop will wait until
*         all 12 coefficients are recieved before storing and implementing them.
*    'v': Verify the coefficients. The control will read and transmit the 12
*         bytes stored in EEPROM.  If they are 'FF', then the control is using
*         the default coefficients.
*    'r': Reset the coefficients.  This loop erases the coefficients stored in
*         EEProm and recalculates the coordinates with the default coefficients.
*
*    This function will not be used on system where the main
*    microprocessor handles the translation equations.  Consequently,
*    this function can be bypassed with the #define SERIAL_RX_SUPPORTED
*    in comm.h.
*    
***************************************************************/

void Process_RX_Data(void)
{
    byte RX_mode;
    unsigned char i;
  
    RX_mode = temp_buffer[0];                   //1st value of the array is the control byte

    switch(RX_mode)
    {
        case empty:   
            break;
       
        case RX_RS_CE:                          //Reset the coefficient values
            
            use_df_coeff = Yes;                 // Set flag to use default coefficients
       
            (void)EraseEEEPage((uchar *)(&EEE_cal_data));  //Erase the stored coefficients in Flash
       
            LCD_aligned = No;                   // Set flag to indicate unscaled data is TX'd
            temp_buffer[0] = 0;                 // Reset the RX buffer
            need_to_recal = Yes;                // Set flag to recalculate the calibration equations       
            stay_awake_for_RX= 0;               // Recieve communication done, allow to go to STOP mode
            break;
       
        case RX_coeff:                          //Implement LCD Alignment coefficients
            
            if (buffer_index == 12)             // 13 bytes of data need to be RX'd before processing
            {
          // ***** CAUTION: You should not write to flash that is not erased.  To be safe
          // ***** Check that the calibration data is erased before writing it.
                if (EEE_cal_data[0] != 0xFF) 
                    EraseEEEPage((uchar *)(&EEE_cal_data));   //Erase Flash memory if not already empty
 
                (void)ProgramEEEBytes((uchar *)(&EEE_cal_data),&temp_buffer[1],12);	  //Store the new calibration coefficients in memory

                LCD_aligned = Yes;              // Set flag to indicated scaled data is TX'd
                use_df_coeff = No;              // Turn off the default coefficient flag
                temp_buffer[0] = 0;             // Reset the RX buffer
                need_to_recal = Yes;            // Set flag to recalculate the calibration equations 
                stay_awake_for_RX= 0;               // Recieve communication done, allow to go to STOP mode
            } else if (buffer_index > 12){      // An error has occured
                temp_buffer[0] = 0;             // Reset the code
                                                // NOTE: You may want to set up a command TO the system processor
                                                //       letting it know it did not correctly recieve the data.
                 stay_awake_for_RX= 0;               // Recieve communication done, allow to go to STOP mode
            }
            break;
            
        case TX_coeff:                          //Transmit the coefficient values currently stored in flash
        
            for (i=0;i<12;i++){
              while(!SCIS1_TDRE);               //Wait for the TX buffer to empty
              SCID = EEE_cal_data[i];           //Send the next byte
            }
            temp_buffer[0] = 0;
            stay_awake_for_RX= 0;               // Recieve communication done, allow to go to STOP mode
            break;
                                                //Reset the RX buffer
        default:
            /* Invalid control byte */ 
            temp_buffer[0] = 0;                 // Reset the RX buffer
            stay_awake_for_RX= 0;               // Recieve communication done, allow to go to STOP mode
            break;
    }
}

